"""
AI Assistant Service на Groq API
Оптимизирован для Bizness Navigator - InnovAItion Hack-2025
ИСПРАВЛЕНА ВЕРСИЯ с обработкой ошибок и rate limits
"""
import json
import time
import logging
from typing import Dict, List, Optional
from functools import wraps
from decouple import config
from groq import Groq

# Настройка логирования
logger = logging.getLogger(__name__)


def rate_limit_handler(func):
    """Декоратор для обработки rate limits Groq API"""
    last_call = {'time': 0}

    @wraps(func)
    def wrapper(self, *args, **kwargs):
        # Минимум 2 секунды между запросами (30 req/min = 2 sec/req)
        elapsed = time.time() - last_call['time']
        if elapsed < 2:
            sleep_time = 2 - elapsed
            logger.debug(f"Rate limiting: sleeping {sleep_time:.2f}s")
            time.sleep(sleep_time)

        try:
            result = func(self, *args, **kwargs)
            last_call['time'] = time.time()
            return result
        except Exception as e:
            error_msg = str(e).lower()
            if "rate_limit" in error_msg or "429" in error_msg:
                logger.warning("⚠️ Rate limit reached, waiting 60 seconds...")
                time.sleep(60)
                return func(self, *args, **kwargs)
            raise

    return wrapper


class GroqAIService:
    """
    AI сервис на базе Groq API
    Бесплатно, быстро, надежно!
    """

    def __init__(self):
        # Используем decouple вместо os.getenv!
        self.api_key = config('GROQ_API_KEY', default='')

        if not self.api_key:
            raise ValueError(
                "❌ GROQ_API_KEY не найден!\n"
                "Установите в .env файле: GROQ_API_KEY=your_key_here\n"
                "Получить ключ: https://console.groq.com"
            )

        try:
            self.client = Groq(api_key=self.api_key)
        except Exception as e:
            raise ValueError(f"❌ Ошибка инициализации Groq client: {e}")

        # Используем самую мощную модель
        self.model = "llama-3.1-70b-versatile"

        # Метрики для мониторинга
        self.metrics = {
            'total_calls': 0,
            'successful_calls': 0,
            'failed_calls': 0,
            'fallback_calls': 0,
        }

        logger.info("✅ Groq AI Service инициализирован")
        print("✅ Groq AI Service инициализирован")

    @rate_limit_handler
    def _call_groq(
            self,
            prompt: str,
            system: str = "",
            temperature: float = 0.7,
            max_tokens: int = 4000
    ) -> str:
        """Базовый вызов Groq API с обработкой ошибок"""

        messages = []
        if system:
            messages.append({"role": "system", "content": system})
        messages.append({"role": "user", "content": prompt})

        self.metrics['total_calls'] += 1

        try:
            logger.debug(f"Calling Groq API with model: {self.model}")

            response = self.client.chat.completions.create(
                model=self.model,
                messages=messages,
                temperature=temperature,
                max_tokens=max_tokens
            )

            self.metrics['successful_calls'] += 1
            logger.debug("✅ Groq API call successful")

            return response.choices[0].message.content

        except Exception as e:
            self.metrics['failed_calls'] += 1
            logger.error(f"❌ Groq API error: {e}")
            raise

    def _parse_json_response(self, response: str) -> Dict:
        """Парсинг JSON из ответа AI с улучшенной обработкой"""
        # Очистка от markdown
        response = response.strip()

        # Удаляем markdown блоки
        if response.startswith("```json"):
            response = response[7:]
        elif response.startswith("```"):
            response = response[3:]
        if response.endswith("```"):
            response = response[:-3]

        response = response.strip()

        try:
            return json.loads(response)
        except json.JSONDecodeError as e:
            logger.error(f"❌ JSON parsing error: {e}")
            logger.debug(f"Response (first 500 chars): {response[:500]}")

            # Пытаемся найти JSON в тексте
            try:
                start = response.find('{')
                end = response.rfind('}') + 1
                if start != -1 and end > start:
                    json_str = response[start:end]
                    return json.loads(json_str)
            except:
                pass

            raise ValueError(f"Не удалось распарсить JSON: {e}")

    def _validate_business_idea(self, data: Dict) -> bool:
        """Валидация структуры бизнес-идеи"""
        required_fields = {
            'title': str,
            'description': str,
            'category': str,
            'investment_required': (int, float),
            'potential_profit': (int, float),
            'payback_period': (int, float),
        }

        for field, expected_type in required_fields.items():
            if field not in data:
                logger.error(f"Missing required field: {field}")
                return False
            if not isinstance(data[field], expected_type):
                logger.error(f"Invalid type for {field}: expected {expected_type}, got {type(data[field])}")
                return False

        # Проверяем разумность значений
        if data['investment_required'] <= 0:
            logger.error("Investment required must be positive")
            return False
        if data['potential_profit'] < 0:
            logger.error("Potential profit cannot be negative")
            return False
        if data['payback_period'] <= 0:
            logger.error("Payback period must be positive")
            return False

        return True

    def generate_business_idea(
            self,
            interests: str,
            budget: float,
            location: str = "Ташкент"
    ) -> Dict:
        """
        🚀 Генерирует бизнес-идею с помощью AI

        Args:
            interests: Интересы пользователя
            budget: Доступный бюджет в сумах
            location: Город/регион

        Returns:
            Dict с полной бизнес-идеей
        """

        logger.info(f"Generating business idea: interests={interests}, budget={budget}, location={location}")

        system = """Ты опытный бизнес-консультант и эксперт по стартапам в Узбекистане.
Твоя задача - создавать реалистичные, инновационные бизнес-идеи с точными финансовыми расчетами.
Учитывай местную специфику, законодательство, менталитет и актуальные тренды 2025 года."""

        prompt = f"""Создай детальную бизнес-идею для предпринимателя в Узбекистане.

📋 ВХОДНЫЕ ДАННЫЕ:
- Интересы/сфера: {interests}
- Доступный бюджет: {budget:,.0f} сум (≈${budget / 12500:.0f})
- Локация: {location}

🎯 ТРЕБОВАНИЯ:
1. Идея должна быть реалистичной и актуальной для Узбекистана 2025
2. Учитывай местные законы, налоги, менталитет
3. Все цифры должны быть точными и обоснованными
4. Срок окупаемости - реалистичный (обычно 6-18 месяцев)
5. Идея должна соответствовать бюджету

📊 ВЕРНИ СТРОГО JSON в следующем формате (без дополнительного текста):

{{
    "title": "Название бизнеса (креативное и запоминающееся)",
    "description": "Детальное описание бизнес-модели на 200-300 слов",
    "category": "Выбери одну: Общепит, Розничная торговля, Услуги, Образование, IT и технологии, Спорт и фитнес",
    "target_audience": "Детальное описание целевой аудитории",
    "investment_required": число_в_сумах,
    "potential_profit": число_месячная_прибыль_в_сумах,
    "payback_period": число_месяцев_окупаемости,
    "risks": ["риск 1", "риск 2", "риск 3"],
    "opportunities": ["возможность 1", "возможность 2"],
    "initial_costs": {{
        "rent": число, "equipment": число, "inventory": число,
        "marketing": число, "licenses": число, "other": число
    }},
    "monthly_costs": {{
        "rent": число, "salaries": число, "utilities": число,
        "supplies": число, "marketing": число, "other": число
    }},
    "revenue_streams": ["источник 1", "источник 2"],
    "competitive_advantages": ["преимущество 1", "преимущество 2"],
    "next_steps": ["шаг 1", "шаг 2", "шаг 3"]
}}

⚠️ ВАЖНО: Верни ТОЛЬКО валидный JSON!"""

        try:
            response = self._call_groq(prompt, system, temperature=0.8)
            data = self._parse_json_response(response)

            # Валидация
            if not self._validate_business_idea(data):
                raise ValueError("Невалидная структура бизнес-идеи")

            logger.info(f"✅ Business idea generated: {data['title']}")
            print(f"✅ Бизнес-идея сгенерирована: {data['title']}")
            return data

        except Exception as e:
            logger.error(f"❌ Error generating business idea: {e}")
            print(f"❌ Ошибка генерации, используем fallback: {e}")
            self.metrics['fallback_calls'] += 1
            return self._generate_fallback_idea(interests, budget, location)

    def analyze_location(
            self,
            address: str,
            business_type: str
    ) -> Dict:
        """
        🗺️ Анализирует локацию для бизнеса
        """

        logger.info(f"Analyzing location: address={address}, type={business_type}")

        system = """Ты эксперт по законодательству Узбекистана в сфере предпринимательства.
Анализируешь локации и даешь конкретные рекомендации."""

        prompt = f"""Проанализируй локацию для открытия бизнеса в Узбекистане.

📍 ДАННЫЕ:
- Адрес: {address}
- Тип бизнеса: {business_type}

📊 ВЕРНИ СТРОГО JSON:

{{
    "status": "approved или conditional или rejected",
    "score": число_от_0_до_100,
    "address": "{address}",
    "coordinates": {{"lat": 41.2995, "lon": 69.2401}},
    "permissions": ["разрешение 1"],
    "restrictions": [
        {{"name": "ограничение", "type": "distance", "description": "описание",
          "object": "объект", "distance": 100, "required_distance": 100}}
    ],
    "requirements": [
        {{"name": "требование", "description": "описание", 
          "estimated_cost": 2000000, "time_days": 30}}
    ],
    "nearby_objects": [
        {{"name": "объект", "type": "school", "distance": 320}}
    ],
    "analysis": "Подробный AI-анализ на 150-200 слов",
    "costs": {{"total": 2000000, "time_days": 30}},
    "recommendations": ["рекомендация 1", "рекомендация 2"]
}}

⚠️ Важные ограничения для Узбекистана:
- Алкоголь: 100м от школ/детсадов
- СЭС: ~500 тыс сум, 15 дней
- Лицензия общепит: ~2 млн сум, 30 дней

Верни ТОЛЬКО JSON!"""

        try:
            response = self._call_groq(prompt, system, temperature=0.5)
            data = self._parse_json_response(response)

            logger.info(f"✅ Location analyzed: {data.get('status', 'unknown')}")
            return data

        except Exception as e:
            logger.error(f"❌ Error analyzing location: {e}")
            self.metrics['fallback_calls'] += 1
            return self._generate_fallback_location_analysis(address, business_type)

    def generate_financial_forecast(
            self,
            business_data: Dict,
            months: int = 12
    ) -> Dict:
        """💰 Генерирует финансовый прогноз"""

        logger.info(f"Generating financial forecast for {months} months")

        system = """Ты финансовый аналитик с опытом работы с малым бизнесом в Узбекистане."""

        prompt = f"""Создай финансовый прогноз для бизнеса.

📊 ДАННЫЕ: {json.dumps(business_data, ensure_ascii=False, indent=2)}

📈 ВЕРНИ JSON с прогнозом на {months} месяцев:

{{
    "months": [
        {{"month": 1, "revenue": число, "costs": число, "profit": число, "cumulative_profit": число}}
    ],
    "summary": {{
        "total_revenue": число,
        "total_costs": число,
        "total_profit": число,
        "average_monthly_profit": число,
        "break_even_month": число,
        "roi_percentage": число
    }},
    "insights": ["инсайт 1", "инсайт 2"]
}}"""

        try:
            response = self._call_groq(prompt, system, temperature=0.3)
            data = self._parse_json_response(response)

            logger.info(f"✅ Forecast generated for {months} months")
            return data

        except Exception as e:
            logger.error(f"❌ Error generating forecast: {e}")
            self.metrics['fallback_calls'] += 1
            return self._generate_fallback_forecast(business_data, months)

    def analyze_market(
            self,
            business_type: str,
            location: str = "Ташкент"
    ) -> Dict:
        """📊 Анализирует рынок"""

        system = """Ты аналитик рынка Узбекистана."""

        prompt = f"""Проанализируй рынок для {business_type} в {location}.

Верни JSON: {{"market_size": "...", "growth_trend": "growing/stable/declining", 
"competition_level": "high/medium/low", "opportunities": [...], "threats": [...]}}"""

        try:
            response = self._call_groq(prompt, system)
            return self._parse_json_response(response)
        except Exception as e:
            logger.error(f"❌ Market analysis error: {e}")
            return {}

    def get_metrics(self) -> Dict:
        """Возвращает метрики использования сервиса"""
        return self.metrics.copy()

    # ===== FALLBACK функции =====

    def _generate_fallback_idea(self, interests: str, budget: float, location: str) -> Dict:
        """Резервная бизнес-идея"""
        logger.warning("Using fallback business idea")
        return {
            "title": f"Стартап в сфере: {interests}",
            "description": f"Инновационный бизнес в {location}, ориентированный на {interests}. Использует современные технологии и учитывает местную специфику.",
            "category": "IT и технологии",
            "target_audience": "Молодежь 18-35 лет",
            "investment_required": float(budget),
            "potential_profit": float(budget * 0.15),
            "payback_period": 12,
            "risks": ["Конкуренция", "Изменение законодательства", "Экономические факторы"],
            "opportunities": ["Рост рынка", "Цифровизация"],
            "initial_costs": {
                "rent": float(budget * 0.2),
                "equipment": float(budget * 0.4),
                "inventory": float(budget * 0.2),
                "marketing": float(budget * 0.1),
                "licenses": float(budget * 0.05),
                "other": float(budget * 0.05)
            },
            "monthly_costs": {
                "rent": float(budget * 0.02),
                "salaries": float(budget * 0.05),
                "utilities": float(budget * 0.01),
                "supplies": float(budget * 0.02),
                "marketing": float(budget * 0.02),
                "other": float(budget * 0.01)
            },
            "revenue_streams": ["Основной продукт/услуга", "Дополнительные услуги"],
            "competitive_advantages": ["Инновационный подход", "Качественный сервис"],
            "next_steps": ["Регистрация бизнеса", "Поиск помещения", "Найм персонала"]
        }

    def _generate_fallback_location_analysis(self, address: str, business_type: str) -> Dict:
        """Резервный анализ локации"""
        logger.warning("Using fallback location analysis")
        return {
            "status": "conditional",
            "score": 70,
            "address": address,
            "coordinates": {"lat": 41.2995, "lon": 69.2401},
            "permissions": ["Коммерческая деятельность разрешена"],
            "restrictions": [],
            "requirements": [
                {
                    "name": "Лицензия",
                    "description": "Необходимо получить лицензию",
                    "estimated_cost": 2000000,
                    "time_days": 30
                }
            ],
            "nearby_objects": [],
            "analysis": f"Локация {address} условно подходит для {business_type}. Требуется дополнительная проверка документов и согласований.",
            "costs": {"total": 2000000, "time_days": 30},
            "recommendations": [
                "Проконсультируйтесь с юристом",
                "Проверьте документы на помещение",
                "Уточните зонирование района"
            ]
        }

    def _generate_fallback_forecast(self, business_data: Dict, months: int) -> Dict:
        """Резервный финансовый прогноз"""
        logger.warning("Using fallback forecast")

        initial_investment = business_data.get('investment_required', 10000000)
        monthly_profit = business_data.get('potential_profit', 1000000)

        months_data = []
        cumulative = -float(initial_investment)

        for month in range(1, months + 1):
            # Постепенный рост
            growth_factor = 0.5 + 0.5 * (month / months)
            revenue = monthly_profit * 1.5 * growth_factor
            costs = monthly_profit * 0.5
            profit = revenue - costs
            cumulative += profit

            months_data.append({
                "month": month,
                "revenue": round(revenue),
                "costs": round(costs),
                "profit": round(profit),
                "cumulative_profit": round(cumulative)
            })

        return {
            "months": months_data,
            "summary": {
                "total_revenue": sum(m['revenue'] for m in months_data),
                "total_costs": sum(m['costs'] for m in months_data),
                "total_profit": sum(m['profit'] for m in months_data),
                "average_monthly_profit": float(monthly_profit),
                "break_even_month": 6,
                "roi_percentage": 50.0
            },
            "insights": [
                "Прогноз показывает постепенный рост выручки",
                "Окупаемость ожидается через 6 месяцев",
                "ROI составляет примерно 50% годовых"
            ]
        }


# Singleton instance
_ai_service = None


def get_ai_service() -> GroqAIService:
    """Получить единственный экземпляр AI сервиса (Singleton pattern)"""
    global _ai_service
    if _ai_service is None:
        _ai_service = GroqAIService()
    return _ai_service