"""
Django settings for Bizness Navigator project.
InnovAItion Hack-2025
"""
import os
from pathlib import Path
from decouple import config as decouple_config

# Build paths
BASE_DIR = Path(__file__).resolve().parent.parent

# Security Settings
SECRET_KEY = decouple_config('SECRET_KEY', default='django-insecure-temporary-key-for-development')
DEBUG = decouple_config('DEBUG', default=True, cast=bool)
ALLOWED_HOSTS = decouple_config('ALLOWED_HOSTS', default='*').split(',')

# Application definition
INSTALLED_APPS = [
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',

    # Third party apps
    'rest_framework',
    'corsheaders',

    # Local apps
    'businesses',
    'locations',
    'ai_assistant',
    'rosetta',
]

MIDDLEWARE = [
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.locale.LocaleMiddleware',  # ⭐ ДОБАВЛЕНО для многоязычности
    'corsheaders.middleware.CorsMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
]

ROOT_URLCONF = 'bizness_navigator.urls'

TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [
            BASE_DIR / 'templates',
            BASE_DIR / 'bizness_navigator' / 'templates',
        ],
        'APP_DIRS': True,
        'OPTIONS': {
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
                'django.template.context_processors.i18n',  # ⭐ ДОБАВЛЕНО для языков
            ],
        },
    },
]

WSGI_APPLICATION = 'bizness_navigator.wsgi.application'

# Database
DATABASES = {
    'default': {
        'ENGINE': 'django.db.backends.sqlite3',
        'NAME': BASE_DIR / 'db.sqlite3',
    }
}

# Password validation
AUTH_PASSWORD_VALIDATORS = [
    {'NAME': 'django.contrib.auth.password_validation.UserAttributeSimilarityValidator'},
    {'NAME': 'django.contrib.auth.password_validation.MinimumLengthValidator'},
    {'NAME': 'django.contrib.auth.password_validation.CommonPasswordValidator'},
    {'NAME': 'django.contrib.auth.password_validation.NumericPasswordValidator'},
]

# ==========================================
# ⭐ INTERNATIONALIZATION (МНОГОЯЗЫЧНОСТЬ)
# ==========================================

# Доступные языки
LANGUAGES = [
    ('en', 'English'),
    ('ru', 'Русский'),
    ('uz', "O'zbekcha"),
]

# Язык по умолчанию
LANGUAGE_CODE = 'ru'  # Русский по умолчанию

# Часовой пояс
TIME_ZONE = 'Asia/Tashkent'

# Включить интернационализацию
USE_I18N = True
USE_L10N = True
USE_TZ = True

# Путь к файлам переводов
LOCALE_PATHS = [
    BASE_DIR / 'locale',
]

# ==========================================
# Static files (CSS, JavaScript, Images)
# ==========================================
STATIC_URL = 'static/'
STATIC_ROOT = BASE_DIR / 'staticfiles'

# Создаем папку static если её нет
STATIC_DIR = BASE_DIR / 'static'
if not STATIC_DIR.exists():
    STATIC_DIR.mkdir(parents=True, exist_ok=True)
    (STATIC_DIR / '.gitkeep').touch()

STATICFILES_DIRS = [STATIC_DIR]

# Media files (User uploads)
MEDIA_URL = 'media/'
MEDIA_ROOT = BASE_DIR / 'media'

# Создаем папку media если её нет
if not MEDIA_ROOT.exists():
    MEDIA_ROOT.mkdir(parents=True, exist_ok=True)
    (MEDIA_ROOT / '.gitkeep').touch()

DEFAULT_AUTO_FIELD = 'django.db.models.BigAutoField'

# CORS settings
CORS_ALLOW_ALL_ORIGINS = DEBUG  # В продакшене установить False

# REST Framework
REST_FRAMEWORK = {
    'DEFAULT_PAGINATION_CLASS': 'rest_framework.pagination.PageNumberPagination',
    'PAGE_SIZE': 10,
    'DEFAULT_RENDERER_CLASSES': [
        'rest_framework.renderers.JSONRenderer',
    ],
}

if DEBUG:
    REST_FRAMEWORK['DEFAULT_RENDERER_CLASSES'].append(
        'rest_framework.renderers.BrowsableAPIRenderer'
    )

# AI Settings
GROQ_API_KEY = decouple_config('GROQ_API_KEY', default='')

# Проверяем наличие ключа в DEBUG режиме
if DEBUG and not GROQ_API_KEY:
    print("⚠️  WARNING: GROQ_API_KEY не установлен!")
    print("   Установите его в .env файле: GROQ_API_KEY=your_key_here")

# Authentication
LOGIN_REDIRECT_URL = '/dashboard/'
LOGOUT_REDIRECT_URL = '/'
LOGIN_URL = '/accounts/login/'

# Session settings
SESSION_COOKIE_AGE = 86400  # 24 часа
SESSION_SAVE_EVERY_REQUEST = False

# Security settings for production
if not DEBUG:
    SECURE_SSL_REDIRECT = True
    SESSION_COOKIE_SECURE = True
    CSRF_COOKIE_SECURE = True
    SECURE_BROWSER_XSS_FILTER = True
    SECURE_CONTENT_TYPE_NOSNIFF = True
    X_FRAME_OPTIONS = 'DENY'

# Logging
LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'verbose': {
            'format': '{levelname} {asctime} {module} {message}',
            'style': '{',
        },
        'simple': {
            'format': '{levelname} {message}',
            'style': '{',
        },
    },
    'handlers': {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'simple' if DEBUG else 'verbose',
        },
        'file': {
            'class': 'logging.FileHandler',
            'filename': BASE_DIR / 'logs' / 'debug.log',
            'formatter': 'verbose',
        },
    },
    'root': {
        'handlers': ['console'],
        'level': 'INFO' if DEBUG else 'WARNING',
    },
    'loggers': {
        'django': {
            'handlers': ['console'],
            'level': 'INFO',
            'propagate': False,
        },
        'businesses': {
            'handlers': ['console', 'file'] if DEBUG else ['console'],
            'level': 'DEBUG' if DEBUG else 'INFO',
            'propagate': False,
        },
        'locations': {
            'handlers': ['console', 'file'] if DEBUG else ['console'],
            'level': 'DEBUG' if DEBUG else 'INFO',
            'propagate': False,
        },
        'ai_assistant': {
            'handlers': ['console', 'file'] if DEBUG else ['console'],
            'level': 'DEBUG' if DEBUG else 'INFO',
            'propagate': False,
        },
    },
}

# Создаем папку для логов
LOGS_DIR = BASE_DIR / 'logs'
if not LOGS_DIR.exists():
    LOGS_DIR.mkdir(parents=True, exist_ok=True)

# ⭐ СОЗДАЕМ ПАПКУ ДЛЯ ПЕРЕВОДОВ
LOCALE_DIR = BASE_DIR / 'locale'
if not LOCALE_DIR.exists():
    LOCALE_DIR.mkdir(parents=True, exist_ok=True)

# Custom settings
MAX_UPLOAD_SIZE = 5242880  # 5MB
BUSINESS_IDEAS_PER_USER = 50  # Максимум идей на пользователя
LOCATION_CHECKS_PER_DAY = 100  # Максимум проверок локаций в день

# Admin customization
ADMIN_SITE_HEADER = "Bizness Navigator Admin"
ADMIN_SITE_TITLE = "Bizness Navigator"
ADMIN_INDEX_TITLE = "Добро пожаловать в панель управления"