"""
Admin configuration for businesses app
"""
from django.contrib import admin
from .models import BusinessCategory, BusinessIdea, Business, FinancialPlan


@admin.register(BusinessCategory)
class BusinessCategoryAdmin(admin.ModelAdmin):
    list_display = ['name', 'icon', 'created_at']
    search_fields = ['name', 'description']
    ordering = ['name']


@admin.register(BusinessIdea)
class BusinessIdeaAdmin(admin.ModelAdmin):
    list_display = ['title', 'user', 'category', 'investment_required', 'potential_profit', 'payback_period',
                    'created_at']
    list_filter = ['category', 'created_at']
    search_fields = ['title', 'description', 'user__username']
    readonly_fields = ['created_at', 'updated_at']

    fieldsets = (
        ('Основная информация', {
            'fields': ('user', 'category', 'title', 'description', 'target_audience')
        }),
        ('Финансы', {
            'fields': ('investment_required', 'potential_profit', 'payback_period')
        }),
        ('AI Данные', {
            'fields': ('market_analysis', 'competitors', 'trends_data', 'ai_confidence'),
            'classes': ('collapse',)
        }),
        ('Метаданные', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )


@admin.register(Business)
class BusinessAdmin(admin.ModelAdmin):
    list_display = ['name', 'user', 'status', 'category', 'initial_capital', 'monthly_profit', 'created_at']
    list_filter = ['status', 'category', 'created_at']
    search_fields = ['name', 'description', 'user__username', 'inn']
    readonly_fields = ['created_at', 'updated_at', 'monthly_profit']

    fieldsets = (
        ('Основная информация', {
            'fields': ('user', 'name', 'description', 'status', 'category', 'idea')
        }),
        ('Финансы', {
            'fields': ('initial_capital', 'monthly_revenue', 'monthly_expenses', 'monthly_profit')
        }),
        ('Юридическая информация', {
            'fields': ('registration_date', 'inn', 'legal_form'),
            'classes': ('collapse',)
        }),
        ('AI Модель', {
            'fields': ('financial_model',),
            'classes': ('collapse',)
        }),
        ('Метаданные', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )

    def monthly_profit(self, obj):
        return f"{obj.monthly_profit:,.0f} сум"

    monthly_profit.short_description = "Месячная прибыль"


@admin.register(FinancialPlan)
class FinancialPlanAdmin(admin.ModelAdmin):
    list_display = ['business', 'period_start', 'period_end', 'projected_profit', 'created_at']
    list_filter = ['created_at', 'period_start']
    search_fields = ['business__name']
    readonly_fields = ['created_at']

    fieldsets = (
        ('Бизнес', {
            'fields': ('business',)
        }),
        ('Период', {
            'fields': ('period_start', 'period_end')
        }),
        ('Прогнозы', {
            'fields': ('projected_revenue', 'projected_expenses', 'projected_profit')
        }),
        ('Детализация', {
            'fields': ('revenue_breakdown', 'expense_breakdown'),
            'classes': ('collapse',)
        }),
        ('AI Рекомендации', {
            'fields': ('ai_recommendations',)
        }),
        ('Метаданные', {
            'fields': ('created_at',),
            'classes': ('collapse',)
        }),
    )