"""
Models for business management - ИСПРАВЛЕННАЯ ВЕРСИЯ
Добавлены все недостающие методы
"""
from django.db import models
from django.contrib.auth.models import User
from decimal import Decimal


class BusinessCategory(models.Model):
    """Категории бизнеса"""
    name = models.CharField(max_length=100, verbose_name="Название")
    description = models.TextField(blank=True)
    icon = models.CharField(max_length=50, blank=True, help_text="Emoji или иконка")

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = "Категория бизнеса"
        verbose_name_plural = "Категории бизнеса"
        ordering = ['name']

    def __str__(self):
        return self.name

    def businesses_count(self):
        """Количество бизнесов в категории"""
        return self.business_set.count()


class BusinessIdea(models.Model):
    """Бизнес-идеи сгенерированные AI"""
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='business_ideas')
    category = models.ForeignKey(BusinessCategory, on_delete=models.SET_NULL, null=True, blank=True)

    # Основная информация
    title = models.CharField(max_length=200, verbose_name="Название")
    description = models.TextField(verbose_name="Описание")
    target_audience = models.TextField(verbose_name="Целевая аудитория")

    # Финансы
    investment_required = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        verbose_name="Требуемые инвестиции"
    )
    potential_profit = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        verbose_name="Потенциальная прибыль/мес"
    )
    payback_period = models.IntegerField(verbose_name="Срок окупаемости (месяцев)")

    # AI данные
    market_analysis = models.JSONField(default=dict, verbose_name="Анализ рынка")
    competitors = models.JSONField(default=list, verbose_name="Конкуренты")
    trends_data = models.JSONField(default=dict, verbose_name="Тренды и аналитика")
    ai_confidence = models.FloatField(default=0.0, verbose_name="Уверенность AI (0-1)")

    # Метаданные
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = "Бизнес-идея"
        verbose_name_plural = "Бизнес-идеи"
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.title} - {self.user.username}"

    @property
    def roi_percentage(self):
        """ROI в процентах (годовой)"""
        if self.investment_required <= 0:
            return 0.0
        annual_profit = float(self.potential_profit) * 12
        return round((annual_profit / float(self.investment_required)) * 100, 2)

    def has_business(self):
        """Проверка, создан ли бизнес из этой идеи"""
        return self.businesses.exists()


class Business(models.Model):
    """Бизнесы пользователя"""
    STATUS_CHOICES = [
        ('planning', 'Планирование'),
        ('registration', 'Регистрация'),
        ('active', 'Активный'),
        ('suspended', 'Приостановлен'),
        ('closed', 'Закрыт'),
    ]

    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='businesses')
    idea = models.ForeignKey(
        BusinessIdea,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='businesses'
    )
    category = models.ForeignKey(BusinessCategory, on_delete=models.SET_NULL, null=True)

    # Основная информация
    name = models.CharField(max_length=200, verbose_name="Название бизнеса")
    description = models.TextField(verbose_name="Описание")
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='planning')

    # Финансы
    initial_capital = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        verbose_name="Начальный капитал"
    )
    monthly_revenue = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        default=0,
        verbose_name="Месячная выручка"
    )
    monthly_expenses = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        default=0,
        verbose_name="Месячные расходы"
    )

    # Дополнительная информация
    registration_date = models.DateField(null=True, blank=True)
    inn = models.CharField(max_length=50, blank=True, verbose_name="ИНН")
    legal_form = models.CharField(max_length=100, blank=True, verbose_name="Форма собственности")

    # AI модель
    financial_model = models.JSONField(default=dict, verbose_name="Финансовая модель")

    # Метаданные
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        verbose_name = "Бизнес"
        verbose_name_plural = "Бизнесы"
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.name} ({self.user.username})"

    @property
    def monthly_profit(self):
        """Месячная прибыль"""
        return self.monthly_revenue - self.monthly_expenses

    @property
    def yearly_profit(self):
        """Годовая прибыль"""
        return self.monthly_profit * 12

    @property
    def roi_percentage(self):
        """ROI в процентах (годовой)"""
        if self.initial_capital <= 0:
            return 0.0
        return round((float(self.yearly_profit) / float(self.initial_capital)) * 100, 2)

    @property
    def break_even_months(self):
        """Срок окупаемости в месяцах"""
        if self.monthly_profit <= 0:
            return float('inf')
        return round(float(self.initial_capital) / float(self.monthly_profit), 2)

    def is_profitable(self):
        """Является ли бизнес прибыльным"""
        return self.monthly_profit > 0

    def get_status_badge_color(self):
        """Цвет бейджа статуса"""
        colors = {
            'planning': 'yellow',
            'registration': 'blue',
            'active': 'green',
            'suspended': 'orange',
            'closed': 'red',
        }
        return colors.get(self.status, 'gray')

    def get_performance_rating(self):
        """Оценка производительности бизнеса (1-5)"""
        roi = self.roi_percentage

        if roi >= 50:
            return 5
        elif roi >= 30:
            return 4
        elif roi >= 15:
            return 3
        elif roi >= 5:
            return 2
        else:
            return 1


class FinancialPlan(models.Model):
    """Финансовый план бизнеса"""
    business = models.ForeignKey(
        Business,
        on_delete=models.CASCADE,
        related_name='financial_plans'
    )

    # Период
    period_start = models.DateField(verbose_name="Начало периода")
    period_end = models.DateField(verbose_name="Конец периода")

    # Прогнозы
    projected_revenue = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        verbose_name="Прогноз выручки"
    )
    projected_expenses = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        verbose_name="Прогноз расходов"
    )
    projected_profit = models.DecimalField(
        max_digits=12,
        decimal_places=2,
        verbose_name="Прогноз прибыли"
    )

    # Детализация
    revenue_breakdown = models.JSONField(default=list, verbose_name="Разбивка доходов")
    expense_breakdown = models.JSONField(default=dict, verbose_name="Разбивка расходов")

    # AI рекомендации
    ai_recommendations = models.TextField(blank=True, verbose_name="Рекомендации AI")

    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = "Финансовый план"
        verbose_name_plural = "Финансовые планы"
        ordering = ['-created_at']

    def __str__(self):
        return f"План для {self.business.name} ({self.period_start})"

    @property
    def duration_months(self):
        """Длительность плана в месяцах"""
        delta = self.period_end - self.period_start
        return round(delta.days / 30)

    @property
    def average_monthly_revenue(self):
        """Средняя месячная выручка"""
        months = self.duration_months
        if months == 0:
            return 0
        return float(self.projected_revenue) / months

    @property
    def profit_margin(self):
        """Маржа прибыли в процентах"""
        if self.projected_revenue <= 0:
            return 0.0
        return round((float(self.projected_profit) / float(self.projected_revenue)) * 100, 2)