"""
Tests for businesses app
"""
from django.test import TestCase, Client
from django.contrib.auth.models import User
from decimal import Decimal
from .models import BusinessCategory, BusinessIdea, Business, FinancialPlan
from datetime import date, timedelta


class BusinessCategoryModelTest(TestCase):
    """Тесты модели BusinessCategory"""

    def setUp(self):
        self.category = BusinessCategory.objects.create(
            name="Тестовая категория",
            description="Описание тестовой категории",
            icon="fa-test"
        )

    def test_category_creation(self):
        """Тест создания категории"""
        self.assertEqual(self.category.name, "Тестовая категория")
        self.assertEqual(str(self.category), "Тестовая категория")

    def test_businesses_count(self):
        """Тест подсчета бизнесов"""
        self.assertEqual(self.category.businesses_count(), 0)


class BusinessIdeaModelTest(TestCase):
    """Тесты модели BusinessIdea"""

    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        self.idea = BusinessIdea.objects.create(
            user=self.user,
            title="Тестовая идея",
            description="Описание идеи",
            target_audience="Тестовая аудитория",
            investment_required=Decimal('10000000'),
            potential_profit=Decimal('1000000'),
            payback_period=10
        )

    def test_idea_creation(self):
        """Тест создания идеи"""
        self.assertEqual(self.idea.title, "Тестовая идея")
        self.assertEqual(str(self.idea), "Тестовая идея - testuser")

    def test_roi_percentage(self):
        """Тест расчета ROI"""
        roi = self.idea.roi_percentage
        # (1,000,000 * 12) / 10,000,000 * 100 = 120%
        self.assertEqual(roi, 120.0)

    def test_has_business(self):
        """Тест проверки наличия бизнеса"""
        self.assertFalse(self.idea.has_business())


class BusinessModelTest(TestCase):
    """Тесты модели Business"""

    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        self.business = Business.objects.create(
            user=self.user,
            name="Тестовый бизнес",
            description="Описание бизнеса",
            status='planning',
            initial_capital=Decimal('15000000'),
            monthly_revenue=Decimal('2000000'),
            monthly_expenses=Decimal('1200000')
        )

    def test_business_creation(self):
        """Тест создания бизнеса"""
        self.assertEqual(self.business.name, "Тестовый бизнес")
        self.assertEqual(str(self.business), "Тестовый бизнес (testuser)")

    def test_monthly_profit(self):
        """Тест расчета месячной прибыли"""
        self.assertEqual(self.business.monthly_profit, Decimal('800000'))

    def test_yearly_profit(self):
        """Тест расчета годовой прибыли"""
        self.assertEqual(self.business.yearly_profit, Decimal('9600000'))

    def test_roi_percentage(self):
        """Тест расчета ROI"""
        # 9,600,000 / 15,000,000 * 100 = 64%
        self.assertAlmostEqual(float(self.business.roi_percentage), 64.0, places=1)

    def test_break_even_months(self):
        """Тест расчета срока окупаемости"""
        # 15,000,000 / 800,000 = 18.75 месяцев
        self.assertAlmostEqual(self.business.break_even_months, 18.75, places=2)

    def test_is_profitable(self):
        """Тест проверки прибыльности"""
        self.assertTrue(self.business.is_profitable())

    def test_get_status_badge_color(self):
        """Тест получения цвета статуса"""
        self.assertEqual(self.business.get_status_badge_color(), 'yellow')

    def test_get_performance_rating(self):
        """Тест оценки производительности"""
        rating = self.business.get_performance_rating()
        self.assertGreaterEqual(rating, 1)
        self.assertLessEqual(rating, 5)


class FinancialPlanModelTest(TestCase):
    """Тесты модели FinancialPlan"""

    def setUp(self):
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        self.business = Business.objects.create(
            user=self.user,
            name="Тестовый бизнес",
            description="Описание",
            initial_capital=Decimal('10000000')
        )
        self.plan = FinancialPlan.objects.create(
            business=self.business,
            period_start=date.today(),
            period_end=date.today() + timedelta(days=365),
            projected_revenue=Decimal('24000000'),
            projected_expenses=Decimal('18000000'),
            projected_profit=Decimal('6000000')
        )

    def test_plan_creation(self):
        """Тест создания плана"""
        self.assertIsNotNone(self.plan)
        self.assertEqual(self.plan.business, self.business)

    def test_duration_months(self):
        """Тест длительности плана"""
        self.assertEqual(self.plan.duration_months, 12)

    def test_average_monthly_revenue(self):
        """Тест средней месячной выручки"""
        # 24,000,000 / 12 = 2,000,000
        self.assertEqual(self.plan.average_monthly_revenue, 2000000.0)

    def test_profit_margin(self):
        """Тест маржи прибыли"""
        # 6,000,000 / 24,000,000 * 100 = 25%
        self.assertEqual(self.plan.profit_margin, 25.0)


class BusinessViewsTest(TestCase):
    """Тесты views"""

    def setUp(self):
        self.client = Client()
        self.user = User.objects.create_user(
            username='testuser',
            password='testpass123'
        )
        self.client.login(username='testuser', password='testpass123')

    def test_dashboard_view(self):
        """Тест дашборда"""
        response = self.client.get('/dashboard/')
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'businesses/dashboard.html')

    def test_generate_idea_view_get(self):
        """Тест GET запроса генерации идеи"""
        response = self.client.get('/dashboard/ideas/generate/')
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'businesses/generate_idea.html')

    def test_business_list_view(self):
        """Тест списка бизнесов"""
        response = self.client.get('/dashboard/businesses/')
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'businesses/business_list.html')