"""
Views for business management
ИСПРАВЛЕНА ВЕРСИЯ с обработкой ошибок
"""
from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.http import JsonResponse
from django.views.decorators.http import require_http_methods
from django.db import transaction
import json
import logging
from datetime import datetime, timedelta
from .models import Business, BusinessIdea, BusinessCategory, FinancialPlan
from ai_assistant.services import get_ai_service
from django.shortcuts import render

logger = logging.getLogger(__name__)


@login_required
def dashboard(request):
    """Главная панель пользователя"""
    try:
        businesses = Business.objects.filter(user=request.user).select_related('category')
        recent_ideas = BusinessIdea.objects.filter(user=request.user).order_by('-created_at')[:5]

        context = {
            'businesses': businesses,
            'recent_ideas': recent_ideas,
            'total_businesses': businesses.count(),
            'active_businesses': businesses.filter(status='active').count()
        }

        return render(request, 'businesses/dashboard.html', context)

    except Exception as e:
        logger.error(f"Dashboard error: {e}")
        messages.error(request, "Ошибка загрузки панели управления")
        return render(request, 'businesses/dashboard.html', {'businesses': [], 'recent_ideas': []})


@login_required
def generate_idea(request):
    """Генерация бизнес-идеи"""
    if request.method == 'POST':
        try:
            # Валидация входных данных
            interests = request.POST.get('interests', '').strip()
            if not interests:
                messages.error(request, "❌ Укажите ваши интересы")
                return redirect('generate_idea')

            # Валидация бюджета
            try:
                budget = float(request.POST.get('budget', 0))
                if budget <= 0:
                    messages.error(request, "❌ Бюджет должен быть больше нуля")
                    return redirect('generate_idea')
                if budget < 1000000:  # Минимум 1 млн сум
                    messages.error(request, "❌ Минимальный бюджет: 1,000,000 сум")
                    return redirect('generate_idea')
            except (ValueError, TypeError):
                messages.error(request, "❌ Неверный формат бюджета")
                return redirect('generate_idea')

            location = request.POST.get('location', 'Ташкент').strip()
            if not location:
                location = 'Ташкент'

            logger.info(f"Generating idea: user={request.user.username}, interests={interests}, budget={budget}")

            # Генерируем идею через ИИ
            ai_service = get_ai_service()
            idea_data = ai_service.generate_business_idea(interests, budget, location)

            # Проверяем обязательные поля
            required_fields = ['title', 'description', 'investment_required', 'potential_profit', 'payback_period']
            for field in required_fields:
                if field not in idea_data:
                    raise ValueError(f"AI не вернул поле: {field}")

            # Определяем категорию
            category = None
            category_mapping = {
                'общепит': ['кафе', 'ресторан', 'бар', 'столовая', 'фастфуд'],
                'розничная торговля': ['магазин', 'супермаркет', 'торговля'],
                'услуги': ['салон', 'парикмахерская', 'сервис'],
                'образование': ['школа', 'курсы', 'обучение', 'тренинг'],
                'it': ['программирование', 'разработка', 'it', 'software'],
                'спорт': ['фитнес', 'спортзал', 'тренажер', 'йога'],
            }

            title_lower = idea_data['title'].lower()
            for cat_name, keywords in category_mapping.items():
                if any(keyword in title_lower for keyword in keywords):
                    category = BusinessCategory.objects.filter(name__icontains=cat_name).first()
                    break

            # Сохраняем идею
            with transaction.atomic():
                idea = BusinessIdea.objects.create(
                    user=request.user,
                    category=category,
                    title=idea_data['title'],
                    description=idea_data['description'],
                    target_audience=idea_data.get('target_audience', ''),
                    investment_required=idea_data['investment_required'],
                    potential_profit=idea_data['potential_profit'],
                    payback_period=idea_data['payback_period'],
                    market_analysis=idea_data.get('market_analysis', {}),
                    competitors=idea_data.get('competitors', []),
                    ai_confidence=0.85,
                    trends_data={
                        'risks': idea_data.get('risks', []),
                        'opportunities': idea_data.get('opportunities', []),
                        'recommendations': idea_data.get('recommendations', []),
                        'initial_costs': idea_data.get('initial_costs', {}),
                        'monthly_costs': idea_data.get('monthly_costs', {}),
                        'revenue_streams': idea_data.get('revenue_streams', []),
                        'competitive_advantages': idea_data.get('competitive_advantages', []),
                        'next_steps': idea_data.get('next_steps', []),
                    }
                )

            logger.info(f"Idea created: id={idea.id}, title={idea.title}")
            messages.success(request, f"✅ Бизнес-идея '{idea.title}' успешно создана!")
            return redirect('business_idea_detail', idea_id=idea.id)

        except ValueError as e:
            logger.error(f"Validation error: {e}")
            messages.error(request, f"❌ Ошибка валидации: {str(e)}")
            return redirect('generate_idea')

        except Exception as e:
            logger.error(f"Error generating idea: {e}", exc_info=True)
            messages.error(request, f"❌ Ошибка генерации идеи. Попробуйте еще раз или измените параметры.")
            return redirect('generate_idea')

    # GET request
    categories = BusinessCategory.objects.all()
    context = {'categories': categories}
    return render(request, 'businesses/generate_idea.html', context)


@login_required
def business_idea_detail(request, idea_id):
    """Детальная информация о бизнес-идее"""
    idea = get_object_or_404(BusinessIdea, id=idea_id, user=request.user)

    # Рассчитываем ROI
    roi_months = 0
    if idea.potential_profit > 0:
        roi_months = round(float(idea.investment_required) / float(idea.potential_profit), 1)

    context = {
        'idea': idea,
        'roi_months': roi_months,
        'roi_years': round(roi_months / 12, 1) if roi_months else 0,
    }

    return render(request, 'businesses/idea_detail.html', context)


@login_required
def create_business_from_idea(request, idea_id):
    """Создание бизнеса из идеи"""
    idea = get_object_or_404(BusinessIdea, id=idea_id, user=request.user)

    if request.method == 'POST':
        try:
            name = request.POST.get('name', '').strip()
            if not name:
                messages.error(request, "❌ Укажите название бизнеса")
                return redirect('create_business_from_idea', idea_id=idea_id)

            # Валидация капитала
            try:
                initial_capital = float(request.POST.get('initial_capital', idea.investment_required))
                if initial_capital <= 0:
                    messages.error(request, "❌ Начальный капитал должен быть больше нуля")
                    return redirect('create_business_from_idea', idea_id=idea_id)
            except (ValueError, TypeError):
                messages.error(request, "❌ Неверный формат капитала")
                return redirect('create_business_from_idea', idea_id=idea_id)

            with transaction.atomic():
                # Создаем бизнес
                business = Business.objects.create(
                    user=request.user,
                    idea=idea,
                    category=idea.category,
                    name=name,
                    description=idea.description,
                    status='planning',
                    initial_capital=initial_capital,
                    financial_model={
                        'investment': float(idea.investment_required),
                        'expected_monthly_profit': float(idea.potential_profit),
                        'payback_period': idea.payback_period
                    }
                )

                # Генерируем финансовый прогноз
                ai_service = get_ai_service()
                forecast_data = ai_service.generate_financial_forecast({
                    'initial_capital': initial_capital,
                    'investment_required': float(idea.investment_required),
                    'potential_profit': float(idea.potential_profit),
                    'business_type': idea.title,
                    'category': idea.category.name if idea.category else 'Общее'
                })

                # Сохраняем прогноз
                start_date = datetime.now().date()

                FinancialPlan.objects.create(
                    business=business,
                    period_start=start_date,
                    period_end=start_date + timedelta(days=365),
                    projected_revenue=forecast_data['summary']['total_revenue'],
                    projected_expenses=forecast_data['summary']['total_costs'],
                    projected_profit=forecast_data['summary']['total_profit'],
                    revenue_breakdown=forecast_data['months'],
                    expense_breakdown={},
                    ai_recommendations='\n'.join(forecast_data.get('insights', []))
                )

            logger.info(f"Business created: id={business.id}, name={business.name}")
            messages.success(request, f"✅ Бизнес '{business.name}' успешно создан!")
            return redirect('business_detail', business_id=business.id)

        except Exception as e:
            logger.error(f"Error creating business: {e}", exc_info=True)
            messages.error(request, f"❌ Ошибка создания бизнеса: {str(e)}")
            return redirect('create_business_from_idea', idea_id=idea_id)

    context = {'idea': idea}
    return render(request, 'businesses/create_from_idea.html', context)


@login_required
def business_list(request):
    """Список бизнесов пользователя"""
    try:
        businesses = Business.objects.filter(user=request.user).select_related('category')

        # Фильтры
        status = request.GET.get('status')
        if status and status in dict(Business.STATUS_CHOICES):
            businesses = businesses.filter(status=status)

        category_id = request.GET.get('category')
        if category_id:
            try:
                businesses = businesses.filter(category_id=int(category_id))
            except (ValueError, TypeError):
                pass

        categories = BusinessCategory.objects.all()

        context = {
            'businesses': businesses,
            'categories': categories,
            'selected_status': status,
            'selected_category': category_id,
            'status_choices': Business.STATUS_CHOICES,
        }

        return render(request, 'businesses/business_list.html', context)

    except Exception as e:
        logger.error(f"Business list error: {e}")
        messages.error(request, "Ошибка загрузки списка бизнесов")
        return render(request, 'businesses/business_list.html', {'businesses': []})


@login_required
def business_detail(request, business_id):
    """Детали бизнеса"""
    business = get_object_or_404(Business, id=business_id, user=request.user)
    financial_plans = business.financial_plans.all()[:5]  # Последние 5 планов

    context = {
        'business': business,
        'financial_plans': financial_plans,
        'latest_plan': financial_plans.first() if financial_plans else None,
    }

    return render(request, 'businesses/business_detail.html', context)


@login_required
@require_http_methods(["POST"])
def update_business_status(request, business_id):
    """
    Обновление статуса бизнеса
    ИСПРАВЛЕННАЯ ВЕРСИЯ с детальной валидацией
    """
    try:
        # Получаем бизнес пользователя
        business = get_object_or_404(Business, id=business_id, user=request.user)

        # Парсим JSON
        try:
            data = json.loads(request.body)
        except json.JSONDecodeError:
            logger.error("Invalid JSON in request body")
            return JsonResponse({
                'success': False,
                'error': 'Неверный формат данных'
            }, status=400)

        new_status = data.get('status')

        # Валидация статуса
        if not new_status:
            return JsonResponse({
                'success': False,
                'error': 'Статус не указан'
            }, status=400)

        # Проверяем, что статус валиден
        valid_statuses = dict(Business.STATUS_CHOICES).keys()
        if new_status not in valid_statuses:
            return JsonResponse({
                'success': False,
                'error': f'Неверный статус. Допустимые: {", ".join(valid_statuses)}'
            }, status=400)

        # Проверяем, не тот ли же статус
        if business.status == new_status:
            return JsonResponse({
                'success': True,
                'status': business.get_status_display(),
                'status_code': business.status,
                'message': 'Статус не изменился'
            })

        # Обновляем статус
        old_status = business.status
        business.status = new_status
        business.save(update_fields=['status', 'updated_at'])

        logger.info(
            f"Business status updated: id={business.id}, "
            f"user={request.user.username}, "
            f"old_status={old_status}, new_status={new_status}"
        )

        return JsonResponse({
            'success': True,
            'status': business.get_status_display(),
            'status_code': business.status,
            'message': f'Статус успешно изменен с "{dict(Business.STATUS_CHOICES)[old_status]}" на "{business.get_status_display()}"'
        })

    except Business.DoesNotExist:
        logger.error(f"Business not found: id={business_id}, user={request.user.username}")
        return JsonResponse({
            'success': False,
            'error': 'Бизнес не найден'
        }, status=404)

    except Exception as e:
        logger.exception(f"Error updating business status: {e}")
        return JsonResponse({
            'success': False,
            'error': f'Ошибка сервера: {str(e)}'
        }, status=500)

@login_required
def financial_forecast(request, business_id):
    """Финансовый прогноз для бизнеса"""
    business = get_object_or_404(Business, id=business_id, user=request.user)
    plans = business.financial_plans.all()[:10]  # Последние 10 планов

    # Подготовка данных для графиков
    chart_data = []
    if plans.exists():
        latest_plan = plans.first()
        monthly_data = latest_plan.revenue_breakdown

        if monthly_data:
            for month_data in monthly_data:
                chart_data.append({
                    'month': f"Месяц {month_data.get('month', 0)}",
                    'revenue': month_data.get('revenue', 0),
                    'costs': month_data.get('costs', 0),  # ← ИСПРАВЛЕНО!
                    'profit': month_data.get('profit', 0)
                })

    context = {
        'business': business,
        'financial_plans': plans,
        'latest_plan': plans.first() if plans else None,
        'chart_data': json.dumps(chart_data, ensure_ascii=False)
    }

    return render(request, 'businesses/financial_forecast.html', context)


@login_required
@require_http_methods(["POST"])
def regenerate_forecast(request, business_id):
    """Пересоздать финансовый прогноз"""
    business = get_object_or_404(Business, id=business_id, user=request.user)

    try:
        # Генерируем новый прогноз
        ai_service = get_ai_service()
        forecast_data = ai_service.generate_financial_forecast({
            'initial_capital': float(business.initial_capital),
            'business_type': business.name,
            'category': business.category.name if business.category else 'Общее',
            'current_revenue': float(business.monthly_revenue) if business.monthly_revenue else 0,
            'current_expenses': float(business.monthly_expenses) if business.monthly_expenses else 0,
        })

        # Сохраняем новый прогноз
        start_date = datetime.now().date()

        with transaction.atomic():
            plan = FinancialPlan.objects.create(
                business=business,
                period_start=start_date,
                period_end=start_date + timedelta(days=365),
                projected_revenue=forecast_data['summary']['total_revenue'],
                projected_expenses=forecast_data['summary']['total_costs'],
                projected_profit=forecast_data['summary']['total_profit'],
                revenue_breakdown=forecast_data['months'],
                expense_breakdown={},
                ai_recommendations='\n'.join(forecast_data.get('insights', []))
            )

        logger.info(f"Forecast regenerated: business_id={business.id}, plan_id={plan.id}")

        return JsonResponse({
            'success': True,
            'message': '✅ Прогноз успешно обновлен',
            'plan_id': plan.id
        })

    except Exception as e:
        logger.error(f"Error regenerating forecast: {e}", exc_info=True)
        return JsonResponse({
            'success': False,
            'error': f'Ошибка обновления прогноза: {str(e)}'
        }, status=500)


def location_search(request):
    return render(request, 'businesses/location_search.html')

@login_required
def user_profile(request):
    """Профиль пользователя"""
    context = {
        'user': request.user,
    }
    return render(request, 'profile.html', context)


from django.contrib.auth import login
from .forms import UserRegistrationForm


def register(request):
    """Регистрация нового пользователя"""
    if request.user.is_authenticated:
        return redirect('dashboard')

    if request.method == 'POST':
        form = UserRegistrationForm(request.POST)
        if form.is_valid():
            try:
                user = form.save()
                login(request, user)
                messages.success(request, f'✅ Добро пожаловать, {user.username}! Ваш аккаунт успешно создан.')
                logger.info(f"New user registered: {user.username}")
                return redirect('dashboard')
            except Exception as e:
                logger.error(f"Registration error: {e}")
                messages.error(request, '❌ Ошибка при создании аккаунта. Попробуйте снова.')
        else:
            for field, errors in form.errors.items():
                for error in errors:
                    messages.error(request, f'❌ {error}')
    else:
        form = UserRegistrationForm()

    return render(request, 'registration/register.html', {'form': form})
