"""
Models for location checking and restrictions - ИСПРАВЛЕННАЯ ВЕРСИЯ
Добавлены все недостающие методы
"""
from django.db import models
from django.contrib.auth.models import User


class ZoneType(models.Model):
    """Типы зон (жилая, коммерческая и т.д.)"""
    name = models.CharField(max_length=100, verbose_name="Название")
    code = models.CharField(max_length=20, unique=True)
    description = models.TextField()
    allowed_business_types = models.JSONField(default=list)

    class Meta:
        verbose_name = "Тип зоны"
        verbose_name_plural = "Типы зон"

    def __str__(self):
        return self.name


class Location(models.Model):
    """Локации для бизнеса"""
    address = models.CharField(max_length=300, verbose_name="Адрес")
    latitude = models.DecimalField(max_digits=10, decimal_places=8)
    longitude = models.DecimalField(max_digits=11, decimal_places=8)

    zone_type = models.ForeignKey(ZoneType, on_delete=models.SET_NULL, null=True)

    # Характеристики помещения
    area = models.FloatField(verbose_name="Площадь (м²)")
    floor = models.IntegerField(verbose_name="Этаж", default=1)
    has_separate_entrance = models.BooleanField(default=False)
    has_parking = models.BooleanField(default=False)
    parking_spaces = models.IntegerField(default=0)

    # Коммуникации
    has_water = models.BooleanField(default=True)
    has_sewage = models.BooleanField(default=True)
    has_electricity = models.BooleanField(default=True)
    has_gas = models.BooleanField(default=False)
    has_ventilation = models.BooleanField(default=False)

    # Цена
    rent_price = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)
    sale_price = models.DecimalField(max_digits=12, decimal_places=2, null=True, blank=True)

    # Статус
    is_available = models.BooleanField(default=True)
    owner_contact = models.CharField(max_length=200, blank=True)

    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = "Локация"
        verbose_name_plural = "Локации"

    def __str__(self):
        return self.address

    @property
    def price_per_sqm(self):
        """Цена за квадратный метр"""
        if not self.rent_price or self.area <= 0:
            return 0
        return round(float(self.rent_price) / self.area, 2)

    @property
    def suitability_score(self):
        """Оценка пригодности локации (0-100)"""
        score = 50  # Базовая оценка

        # Коммуникации (+30)
        if self.has_water: score += 5
        if self.has_sewage: score += 5
        if self.has_electricity: score += 10
        if self.has_gas: score += 5
        if self.has_ventilation: score += 5

        # Удобства (+20)
        if self.has_separate_entrance: score += 10
        if self.has_parking: score += 10

        return min(score, 100)


class Restriction(models.Model):
    """Ограничения для бизнеса"""
    RESTRICTION_TYPES = [
        ('distance', 'Дистанционное ограничение'),
        ('zone', 'Зональное ограничение'),
        ('license', 'Лицензионное требование'),
        ('sanitary', 'Санитарное требование'),
        ('fire', 'Пожарное требование'),
    ]

    name = models.CharField(max_length=200, verbose_name="Название")
    restriction_type = models.CharField(max_length=20, choices=RESTRICTION_TYPES)
    description = models.TextField(verbose_name="Описание")

    # Для дистанционных ограничений
    restricted_object_type = models.CharField(max_length=100, blank=True)
    minimum_distance = models.IntegerField(null=True, blank=True, help_text="Минимальная дистанция в метрах")

    # Применимые категории бизнеса
    affected_categories = models.JSONField(default=list)

    # Законодательная база
    legal_reference = models.TextField(blank=True)

    is_active = models.BooleanField(default=True)

    class Meta:
        verbose_name = "Ограничение"
        verbose_name_plural = "Ограничения"

    def __str__(self):
        return self.name

    def applies_to_business(self, business_type):
        """Проверяет, применимо ли ограничение к типу бизнеса"""
        if not self.affected_categories:
            return True  # Если не указаны категории, применяется ко всем

        business_type_lower = business_type.lower()

        # Проверяем точное совпадение или вхождение
        for category in self.affected_categories:
            if category.lower() in business_type_lower or business_type_lower in category.lower():
                return True

        return False


class LocationCheck(models.Model):
    """История проверок локаций"""
    STATUS_CHOICES = [
        ('approved', 'Одобрено'),
        ('conditional', 'С условиями'),
        ('rejected', 'Отклонено'),
    ]

    user = models.ForeignKey(User, on_delete=models.CASCADE)
    location = models.ForeignKey(Location, on_delete=models.CASCADE, null=True, blank=True)

    # Параметры проверки
    address = models.CharField(max_length=300)
    business_type = models.CharField(max_length=100)
    latitude = models.DecimalField(max_digits=10, decimal_places=8, null=True)
    longitude = models.DecimalField(max_digits=11, decimal_places=8, null=True)

    # Результаты проверки
    status = models.CharField(max_length=20, choices=STATUS_CHOICES)
    restrictions_found = models.JSONField(default=list)
    requirements = models.JSONField(default=list)
    nearby_objects = models.JSONField(default=list)

    # ИИ анализ
    ai_analysis = models.TextField(verbose_name="ИИ анализ")
    suitability_score = models.FloatField(default=0.0, verbose_name="Оценка пригодности")

    # Стоимость легализации
    estimated_cost = models.DecimalField(max_digits=10, decimal_places=2, default=0)
    estimated_time_days = models.IntegerField(default=0)

    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = "Проверка локации"
        verbose_name_plural = "Проверки локаций"
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.business_type} - {self.address}"

    @property
    def color_status(self):
        """Цвет статуса"""
        colors = {
            'approved': 'green',
            'conditional': 'orange',
            'rejected': 'red',
        }
        return colors.get(self.status, 'gray')

    @property
    def has_critical_restrictions(self):
        """Есть ли критические ограничения"""
        for restriction in self.restrictions_found:
            if restriction.get('severity') == 'high':
                return True
        return False

    @property
    def is_viable(self):
        """Является ли локация жизнеспособной"""
        return self.status in ['approved', 'conditional'] and self.suitability_score >= 50


class NearbyObject(models.Model):
    """Близлежащие объекты (школы, больницы и т.д.)"""
    OBJECT_TYPES = [
        ('school', 'Школа'),
        ('kindergarten', 'Детский сад'),
        ('hospital', 'Больница'),
        ('mosque', 'Мечеть'),
        ('park', 'Парк'),
        ('market', 'Рынок'),
        ('transport', 'Остановка'),
    ]

    name = models.CharField(max_length=200)
    object_type = models.CharField(max_length=20, choices=OBJECT_TYPES)
    address = models.CharField(max_length=300)
    latitude = models.DecimalField(max_digits=10, decimal_places=8)
    longitude = models.DecimalField(max_digits=11, decimal_places=8)

    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = "Близлежащий объект"
        verbose_name_plural = "Близлежащие объекты"

    def __str__(self):
        return f"{self.get_object_type_display()} - {self.name}"