"""
Geolocation Service с OpenStreetMap (ПОЛНОСТЬЮ БЕСПЛАТНО!)
Замена Yandex Maps
"""
import requests
import math
import time


class GeolocationService:
    """Сервис для работы с геоданными через OpenStreetMap (Nominatim)"""

    def __init__(self):
        # User-Agent обязателен для Nominatim API
        self.user_agent = "BiznessNavigator/1.0 (InnovAItion-Hack-2025; bizness@navigator.uz)"
        self.nominatim_url = "https://nominatim.openstreetmap.org"
        self.overpass_url = "https://overpass-api.de/api/interpreter"

        # Кэш для предотвращения повторных запросов
        self._geocode_cache = {}

    def geocode_address(self, address: str) -> dict:
        """
        Геокодирование адреса через Nominatim (OpenStreetMap)

        Args:
            address: Адрес для поиска

        Returns:
            dict с координатами и адресом
        """
        # Проверяем кэш
        if address in self._geocode_cache:
            return self._geocode_cache[address]

        try:
            # Nominatim API требует задержку между запросами (1 запрос/сек)
            time.sleep(1)

            url = f"{self.nominatim_url}/search"
            params = {
                'q': f"{address}, Uzbekistan",  # Добавляем страну для точности
                'format': 'json',
                'limit': 1,
                'countrycodes': 'uz',  # Ограничиваем Узбекистаном
                'addressdetails': 1
            }

            headers = {
                'User-Agent': self.user_agent,
                'Accept-Language': 'ru'
            }

            response = requests.get(url, params=params, headers=headers, timeout=10)

            if response.status_code == 200:
                data = response.json()

                if data and len(data) > 0:
                    result = {
                        'latitude': float(data[0]['lat']),
                        'longitude': float(data[0]['lon']),
                        'formatted_address': data[0]['display_name'],
                        'success': True
                    }

                    # Кэшируем результат
                    self._geocode_cache[address] = result
                    return result

            # Если не нашли точный адрес, возвращаем центр Ташкента
            print(f"⚠️ Адрес не найден: {address}, используем центр Ташкента")
            return self._get_tashkent_center(address)

        except requests.RequestException as e:
            print(f"❌ Ошибка геокодирования: {e}")
            return self._get_tashkent_center(address)
        except Exception as e:
            print(f"❌ Неожиданная ошибка: {e}")
            return self._get_tashkent_center(address)

    def _get_tashkent_center(self, address: str) -> dict:
        """Возвращает координаты центра Ташкента (fallback)"""
        return {
            'latitude': 41.2995,
            'longitude': 69.2401,
            'formatted_address': f"{address}, Ташкент, Узбекистан",
            'success': False
        }

    def calculate_distance(self, lat1: float, lon1: float, lat2: float, lon2: float) -> float:
        """
        Рассчитывает расстояние между двумя точками в метрах (формула Haversine)

        Args:
            lat1, lon1: Координаты первой точки
            lat2, lon2: Координаты второй точки

        Returns:
            float: Расстояние в метрах
        """
        R = 6371000  # Радиус Земли в метрах

        lat1_rad = math.radians(lat1)
        lon1_rad = math.radians(lon1)
        lat2_rad = math.radians(lat2)
        lon2_rad = math.radians(lon2)

        dlat = lat2_rad - lat1_rad
        dlon = lon2_rad - lon1_rad

        a = (math.sin(dlat / 2) ** 2 +
             math.cos(lat1_rad) * math.cos(lat2_rad) * math.sin(dlon / 2) ** 2)
        c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))

        return R * c

    def find_nearby_objects(self, latitude: float, longitude: float, radius: int = 500) -> list:
        """
        Поиск ближайших объектов через Overpass API (OpenStreetMap)

        Args:
            latitude: Широта
            longitude: Долгота
            radius: Радиус поиска в метрах (по умолчанию 500м)

        Returns:
            list: Список найденных объектов
        """
        try:
            # Overpass API запрос
            overpass_query = f"""
            [out:json][timeout:25];
            (
              node["amenity"~"school|kindergarten|hospital|clinic|pharmacy|place_of_worship|marketplace|bank"]
                (around:{radius},{latitude},{longitude});
              way["amenity"~"school|kindergarten|hospital|clinic|pharmacy|place_of_worship|marketplace|bank"]  
                (around:{radius},{latitude},{longitude});
            );
            out center;
            """

            response = requests.post(
                self.overpass_url,
                data=overpass_query,
                timeout=30,
                headers={'User-Agent': self.user_agent}
            )

            if response.status_code == 200:
                data = response.json()
                objects = []

                for element in data.get('elements', []):
                    tags = element.get('tags', {})
                    amenity = tags.get('amenity', '')
                    name = tags.get('name', tags.get('name:ru', 'Неизвестно'))

                    # Определяем тип объекта
                    obj_type = self._map_amenity_to_type(amenity)

                    # Получаем координаты
                    if 'lat' in element and 'lon' in element:
                        obj_lat = element['lat']
                        obj_lon = element['lon']
                    elif 'center' in element:
                        obj_lat = element['center']['lat']
                        obj_lon = element['center']['lon']
                    else:
                        continue

                    # Рассчитываем расстояние
                    distance = self.calculate_distance(
                        latitude, longitude,
                        obj_lat, obj_lon
                    )

                    if distance <= radius:
                        objects.append({
                            'name': name,
                            'type': obj_type,
                            'distance': int(distance),
                            'amenity': amenity,
                            'latitude': obj_lat,
                            'longitude': obj_lon
                        })

                # Сортируем по расстоянию
                objects.sort(key=lambda x: x['distance'])

                # Если нашли объекты - возвращаем
                if objects:
                    print(f"✅ Найдено {len(objects)} объектов рядом с локацией")
                    return objects[:20]  # Максимум 20 объектов

        except requests.RequestException as e:
            print(f"⚠️ Ошибка Overpass API: {e}")
        except Exception as e:
            print(f"⚠️ Ошибка поиска объектов: {e}")

        # Fallback - возвращаем демо данные
        print("⚠️ Используем демо данные для близлежащих объектов")
        return self._get_demo_nearby_objects()

    def _map_amenity_to_type(self, amenity: str) -> str:
        """Маппинг типов объектов OSM на наши категории"""
        mapping = {
            'school': 'school',
            'kindergarten': 'kindergarten',
            'hospital': 'hospital',
            'clinic': 'hospital',
            'pharmacy': 'pharmacy',
            'place_of_worship': 'mosque',
            'marketplace': 'market',
            'bank': 'bank',
            'bus_station': 'transport',
            'parking': 'parking'
        }
        return mapping.get(amenity, 'other')

    def _get_demo_nearby_objects(self) -> list:
        """Демо данные для fallback (если API не работает)"""
        return [
            {
                'name': 'Школа №25',
                'type': 'school',
                'distance': 320,
                'amenity': 'school'
            },
            {
                'name': 'Детский сад "Болажон"',
                'type': 'kindergarten',
                'distance': 85,
                'amenity': 'kindergarten'
            },
            {
                'name': 'Поликлиника №3',
                'type': 'hospital',
                'distance': 450,
                'amenity': 'clinic'
            },
            {
                'name': 'Мечеть Минор',
                'type': 'mosque',
                'distance': 280,
                'amenity': 'place_of_worship'
            },
            {
                'name': 'Остановка автобуса',
                'type': 'transport',
                'distance': 150,
                'amenity': 'bus_station'
            },
            {
                'name': 'Супермаркет "Макро"',
                'type': 'market',
                'distance': 400,
                'amenity': 'marketplace'
            },
            {
                'name': 'Аптека №12',
                'type': 'pharmacy',
                'distance': 200,
                'amenity': 'pharmacy'
            }
        ]

    def reverse_geocode(self, latitude: float, longitude: float) -> dict:
        """
        Обратное геокодирование (из координат в адрес)

        Args:
            latitude: Широта
            longitude: Долгота

        Returns:
            dict с информацией об адресе
        """
        try:
            time.sleep(1)  # Уважаем лимиты Nominatim

            url = f"{self.nominatim_url}/reverse"
            params = {
                'lat': latitude,
                'lon': longitude,
                'format': 'json',
                'addressdetails': 1,
                'accept-language': 'ru'
            }

            headers = {'User-Agent': self.user_agent}

            response = requests.get(url, params=params, headers=headers, timeout=10)

            if response.status_code == 200:
                data = response.json()
                return {
                    'address': data.get('display_name', 'Неизвестный адрес'),
                    'success': True
                }

        except Exception as e:
            print(f"❌ Ошибка обратного геокодирования: {e}")

        return {
            'address': f"Координаты: {latitude:.4f}, {longitude:.4f}",
            'success': False
        }


# Singleton инстанс для использования в проекте
geolocation_service = GeolocationService()


# Вспомогательные функции

def get_tashkent_regions() -> list:
    """Возвращает список районов Ташкента для выбора"""
    return [
        {'name': 'Мирабадский', 'center': [41.2847, 69.2513]},
        {'name': 'Мирзо-Улугбекский', 'center': [41.3144, 69.3344]},
        {'name': 'Сергелийский', 'center': [41.2197, 69.2239]},
        {'name': 'Учтепинский', 'center': [41.2831, 69.1953]},
        {'name': 'Шайхантахурский', 'center': [41.3253, 69.2775]},
        {'name': 'Яккасарайский', 'center': [41.2886, 69.2286]},
        {'name': 'Юнусабадский', 'center': [41.3334, 69.2892]},
        {'name': 'Алмазарский', 'center': [41.3558, 69.2094]},
        {'name': 'Бектемирский', 'center': [41.2089, 69.3342]},
        {'name': 'Чиланзарский', 'center': [41.2764, 69.2047]},
        {'name': 'Яшнабадский', 'center': [41.2458, 69.2228]},
    ]


def get_popular_addresses() -> list:
    """Возвращает список популярных адресов для быстрого тестирования"""
    return [
        "Улица Навои, 15, Ташкент",
        "Улица Мирабад, 20, Ташкент",
        "Улица Афросиаб, 10, Ташкент",
        "Улица Бухоро, 5, Ташкент",
        "Улица Хадра, 8, Ташкент",
        "Площадь Амира Темура, Ташкент",
        "Чорсу, Ташкент",
        "Олой базар, Ташкент",
    ]