"""
Signals for locations app
Автоматические действия при создании/обновлении объектов
"""
from django.db.models.signals import post_save, pre_save
from django.dispatch import receiver
from django.core.cache import cache
import logging

from .models import LocationCheck, Location, Restriction, NearbyObject

logger = logging.getLogger(__name__)


@receiver(post_save, sender=LocationCheck)
def invalidate_location_cache(sender, instance, created, **kwargs):
    """
    Инвалидирует кэш при создании новой проверки локации
    """
    if created:
        # Очищаем кэш проверок пользователя
        cache_key = f"location_check_{instance.user.id}_{instance.address}_{instance.business_type}"
        cache.delete(cache_key)

        logger.info(f"Cache invalidated for location check: {instance.id}")


@receiver(post_save, sender=LocationCheck)
def log_location_check(sender, instance, created, **kwargs):
    """
    Логирование создания проверки локации
    """
    if created:
        logger.info(
            f"New location check created: ID={instance.id}, "
            f"User={instance.user.username}, "
            f"Type={instance.business_type}, "
            f"Status={instance.status}, "
            f"Score={instance.suitability_score}"
        )


@receiver(pre_save, sender=Location)
def validate_location_coordinates(sender, instance, **kwargs):
    """
    Валидация координат перед сохранением локации
    """
    if instance.latitude < -90 or instance.latitude > 90:
        raise ValueError("Широта должна быть в диапазоне от -90 до 90")

    if instance.longitude < -180 or instance.longitude > 180:
        raise ValueError("Долгота должна быть в диапазоне от -180 до 180")


@receiver(post_save, sender=Location)
def invalidate_location_map_cache(sender, instance, **kwargs):
    """
    Инвалидирует кэш карты при изменении локаций
    """
    # Очищаем кэш страницы с картой
    cache.delete('location_map_geojson')
    logger.info(f"Location map cache invalidated due to change in location: {instance.id}")


@receiver(post_save, sender=Restriction)
def log_restriction_changes(sender, instance, created, **kwargs):
    """
    Логирование изменений ограничений
    """
    action = "created" if created else "updated"
    logger.info(
        f"Restriction {action}: ID={instance.id}, "
        f"Name={instance.name}, "
        f"Type={instance.restriction_type}, "
        f"Active={instance.is_active}"
    )


@receiver(post_save, sender=NearbyObject)
def invalidate_nearby_objects_cache(sender, instance, **kwargs):
    """
    Инвалидирует кэш близлежащих объектов
    """
    # Очищаем кэш в радиусе координат объекта
    cache_pattern = f"nearby_objects_{instance.latitude}_{instance.longitude}_*"
    # Note: для реальной инвалидации паттерна нужен Redis
    logger.info(f"Nearby objects cache should be invalidated for: {instance.id}")