"""
Views for location checking functionality
УЛУЧШЕНИЯ:
- Добавлен кэш для ускорения
- Улучшена обработка ошибок
- Добавлена пагинация
- Оптимизированы запросы к БД
- Добавлены метрики и логирование
"""
from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.http import JsonResponse
from django.views.decorators.http import require_http_methods
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.db.models import Q, Count, Avg
from django.core.cache import cache
from django.views.decorators.cache import cache_page
import json
import logging

from .models import Location, LocationCheck, NearbyObject, Restriction
from .services import geolocation_service
from ai_assistant.services import get_ai_service

logger = logging.getLogger(__name__)


@login_required
def location_search(request):
    """Страница поиска локации"""
    # Статистика для отображения
    stats = {
        'total_checks': LocationCheck.objects.filter(user=request.user).count(),
        'approved_count': LocationCheck.objects.filter(user=request.user, status='approved').count(),
        'avg_score': LocationCheck.objects.filter(user=request.user).aggregate(Avg('suitability_score'))[
                         'suitability_score__avg'] or 0
    }

    context = {
        'stats': stats
    }
    return render(request, 'locations/search.html', context)


@login_required
@require_http_methods(["POST"])
def check_location_api(request):
    """
    API для проверки локации
    УЛУЧШЕНИЯ:
    - Валидация входных данных
    - Детальное логирование
    - Обработка edge cases
    """
    try:
        data = json.loads(request.body)
        address = data.get('address', '').strip()
        business_type = data.get('business_type', '').strip()

        # Валидация
        if not address or len(address) < 5:
            return JsonResponse({
                'error': 'Адрес слишком короткий (минимум 5 символов)'
            }, status=400)

        if not business_type:
            return JsonResponse({
                'error': 'Требуется указать тип бизнеса'
            }, status=400)

        logger.info(f"Location check request: {address} for {business_type} by user {request.user.username}")

        # Проверяем, была ли уже такая проверка недавно (кэш)
        cache_key = f"location_check_{request.user.id}_{address}_{business_type}"
        cached_result = cache.get(cache_key)

        if cached_result:
            logger.info(f"Returning cached result for {cache_key}")
            return JsonResponse({
                'success': True,
                'check_id': cached_result['check_id'],
                'result': cached_result['result'],
                'cached': True
            })

        # Получаем координаты через OpenStreetMap
        coords = geolocation_service.geocode_address(address)

        if not coords.get('success'):
            logger.warning(f"Geocoding failed for address: {address}")

        latitude = coords['latitude']
        longitude = coords['longitude']

        # ИИ проверка локации
        ai_service = get_ai_service()

        try:
            ai_result = ai_service.analyze_location(address, business_type)
        except Exception as e:
            logger.error(f"AI analysis failed: {e}")
            return JsonResponse({
                'error': 'Ошибка при анализе локации. Попробуйте позже.'
            }, status=500)

        # Поиск близлежащих объектов
        nearby = geolocation_service.find_nearby_objects(latitude, longitude, radius=500)
        logger.info(f"Found {len(nearby)} nearby objects")

        # Проверяем ограничения из базы
        restrictions = Restriction.objects.filter(is_active=True).select_related()
        found_restrictions = []

        for restriction in restrictions:
            # Проверяем применимость к типу бизнеса
            if not restriction.applies_to_business(business_type):
                continue

            if restriction.restriction_type == 'distance':
                # Проверяем дистанцию до объектов
                for obj in nearby:
                    if obj['type'] == restriction.restricted_object_type or \
                            obj.get('amenity') == restriction.restricted_object_type:

                        if obj['distance'] < restriction.minimum_distance:
                            found_restrictions.append({
                                'name': restriction.name,
                                'type': restriction.restriction_type,
                                'description': restriction.description,
                                'object': obj['name'],
                                'distance': obj['distance'],
                                'required_distance': restriction.minimum_distance,
                                'severity': 'high' if obj[
                                                          'distance'] < restriction.minimum_distance * 0.5 else 'medium',
                                'legal_reference': restriction.legal_reference
                            })
                            logger.info(f"Found restriction: {restriction.name} for object {obj['name']}")

        # Сохраняем проверку
        try:
            location_check = LocationCheck.objects.create(
                user=request.user,
                address=address,
                business_type=business_type,
                latitude=latitude,
                longitude=longitude,
                status=ai_result['status'],
                restrictions_found=found_restrictions,
                requirements=ai_result['requirements'],
                nearby_objects=nearby,
                ai_analysis=ai_result['analysis'],
                suitability_score=ai_result['score'],
                estimated_cost=ai_result['costs']['total'],
                estimated_time_days=ai_result['costs']['time_days']
            )
            logger.info(f"Location check saved with ID: {location_check.id}")
        except Exception as e:
            logger.error(f"Failed to save location check: {e}")
            return JsonResponse({
                'error': 'Ошибка при сохранении результатов'
            }, status=500)

        # Формируем ответ
        result = {
            'status': ai_result['status'],
            'score': ai_result['score'],
            'address': coords['formatted_address'],
            'coordinates': {
                'lat': float(latitude),
                'lon': float(longitude)
            },
            'permissions': ai_result['permissions'],
            'restrictions': found_restrictions,
            'requirements': ai_result['requirements'],
            'nearby_objects': nearby,
            'analysis': ai_result['analysis'],
            'costs': {
                'total': float(ai_result['costs']['total']),
                'time_days': ai_result['costs']['time_days'],
                'breakdown': ai_result['requirements']
            },
            'recommendations': ai_result['recommendations']
        }

        # Кэшируем результат на 1 час
        cache_data = {
            'check_id': location_check.id,
            'result': result
        }
        cache.set(cache_key, cache_data, 3600)

        return JsonResponse({
            'success': True,
            'check_id': location_check.id,
            'result': result
        })

    except json.JSONDecodeError:
        logger.error("Invalid JSON in request body")
        return JsonResponse({
            'error': 'Неверный формат данных'
        }, status=400)
    except Exception as e:
        logger.exception(f"Unexpected error in check_location_api: {e}")
        return JsonResponse({
            'error': f'Непредвиденная ошибка: {str(e)}'
        }, status=500)


@login_required
def location_check_detail(request, check_id):
    """Детальная информация о проверке локации"""
    try:
        check = get_object_or_404(
            LocationCheck.objects.select_related('user', 'location'),
            id=check_id,
            user=request.user
        )

        context = {
            'check': check,
            'status_color': check.color_status,
            'has_critical_restrictions': check.has_critical_restrictions,
            'is_viable': check.is_viable
        }

        return render(request, 'locations/check_detail.html', context)

    except Exception as e:
        logger.error(f"Error in location_check_detail: {e}")
        return redirect('location_search')


@login_required
def location_history(request):
    """
    История проверок локаций с фильтрацией и пагинацией
    """
    # Базовый queryset с оптимизацией
    checks_list = LocationCheck.objects.filter(user=request.user).select_related(
        'location', 'location__zone_type'
    ).order_by('-created_at')

    # Фильтрация
    status_filter = request.GET.get('status')
    business_type_filter = request.GET.get('business_type')

    if status_filter:
        checks_list = checks_list.filter(status=status_filter)

    if business_type_filter:
        checks_list = checks_list.filter(business_type__icontains=business_type_filter)

    # Статистика
    total_count = checks_list.count()
    avg_score = checks_list.aggregate(Avg('suitability_score'))['suitability_score__avg'] or 0

    # Пагинация
    paginator = Paginator(checks_list, 10)  # 10 проверок на страницу
    page = request.GET.get('page', 1)

    try:
        checks = paginator.page(page)
    except PageNotAnInteger:
        checks = paginator.page(1)
    except EmptyPage:
        checks = paginator.page(paginator.num_pages)

    # Уникальные типы бизнеса для фильтра
    business_types = LocationCheck.objects.filter(
        user=request.user
    ).values_list('business_type', flat=True).distinct()

    context = {
        'checks': checks,
        'business_types': business_types,
        'status_filter': status_filter,
        'business_type_filter': business_type_filter,
        'total_count': total_count,
        'avg_score': round(avg_score, 1)
    }

    return render(request, 'locations/history.html', context)


@login_required
@cache_page(60 * 15)  # Кэш на 15 минут
def location_map(request):
    """
    Карта с локациями
    УЛУЧШЕНИЯ: кэширование и оптимизация запросов
    """
    locations = Location.objects.filter(is_available=True).select_related('zone_type')

    # Преобразуем в GeoJSON
    features = []
    for loc in locations:
        features.append({
            'type': 'Feature',
            'geometry': {
                'type': 'Point',
                'coordinates': [float(loc.longitude), float(loc.latitude)]
            },
            'properties': {
                'id': loc.id,
                'address': loc.address,
                'area': loc.area,
                'rent_price': float(loc.rent_price) if loc.rent_price else None,
                'zone_type': loc.zone_type.name if loc.zone_type else None,
                'suitability_score': loc.suitability_score,
                'has_parking': loc.has_parking,
                'has_ventilation': loc.has_ventilation
            }
        })

    geojson = {
        'type': 'FeatureCollection',
        'features': features
    }

    context = {
        'locations_geojson': json.dumps(geojson),
        'total_locations': len(features)
    }

    return render(request, 'locations/map.html', context)


@login_required
@require_http_methods(["GET"])
def nearby_suggestions_api(request):
    """
    API для получения альтернативных локаций
    УЛУЧШЕНИЯ: добавлена сортировка по пригодности
    """
    lat = request.GET.get('lat')
    lon = request.GET.get('lon')
    business_type = request.GET.get('business_type')

    if not all([lat, lon, business_type]):
        return JsonResponse({
            'error': 'Требуются координаты и тип бизнеса'
        }, status=400)

    try:
        lat = float(lat)
        lon = float(lon)

        # Проверяем кэш
        cache_key = f"nearby_suggestions_{lat}_{lon}_{business_type}"
        cached_result = cache.get(cache_key)

        if cached_result:
            return JsonResponse(cached_result)

        # Ищем подходящие локации в радиусе 2км
        nearby_locations = []

        # Оптимизированный запрос
        all_locations = Location.objects.filter(
            is_available=True
        ).select_related('zone_type')

        for location in all_locations:
            distance = geolocation_service.calculate_distance(
                lat, lon,
                float(location.latitude),
                float(location.longitude)
            )

            if distance <= 2000:  # 2км
                nearby_locations.append({
                    'id': location.id,
                    'address': location.address,
                    'distance': round(distance, 0),
                    'area': location.area,
                    'rent_price': float(location.rent_price) if location.rent_price else None,
                    'price_per_sqm': float(location.price_per_sqm),
                    'coordinates': {
                        'lat': float(location.latitude),
                        'lon': float(location.longitude)
                    },
                    'features': {
                        'separate_entrance': location.has_separate_entrance,
                        'parking': location.has_parking,
                        'ventilation': location.has_ventilation,
                        'floor': location.floor
                    },
                    'zone_type': location.zone_type.name if location.zone_type else None,
                    'suitability_score': location.suitability_score
                })

        # Сортируем: сначала по пригодности, потом по расстоянию
        nearby_locations.sort(key=lambda x: (-x['suitability_score'], x['distance']))

        result = {
            'success': True,
            'locations': nearby_locations[:10],  # Топ 10
            'total_found': len(nearby_locations)
        }

        # Кэшируем на 30 минут
        cache.set(cache_key, result, 1800)

        return JsonResponse(result)

    except ValueError:
        return JsonResponse({
            'error': 'Неверный формат координат'
        }, status=400)
    except Exception as e:
        logger.exception(f"Error in nearby_suggestions_api: {e}")
        return JsonResponse({
            'error': str(e)
        }, status=500)


@login_required
@require_http_methods(["DELETE"])
def delete_location_check(request, check_id):
    """Удаление проверки локации"""
    try:
        check = get_object_or_404(LocationCheck, id=check_id, user=request.user)
        check.delete()

        logger.info(f"Location check {check_id} deleted by user {request.user.username}")

        return JsonResponse({
            'success': True,
            'message': 'Проверка удалена'
        })
    except Exception as e:
        logger.error(f"Error deleting location check: {e}")
        return JsonResponse({
            'error': 'Ошибка при удалении'
        }, status=500)


@login_required
def location_statistics(request):
    """Страница со статистикой проверок"""
    checks = LocationCheck.objects.filter(user=request.user)

    # Агрегированная статистика
    stats = {
        'total': checks.count(),
        'by_status': {
            'approved': checks.filter(status='approved').count(),
            'conditional': checks.filter(status='conditional').count(),
            'rejected': checks.filter(status='rejected').count(),
        },
        'by_business_type': checks.values('business_type').annotate(
            count=Count('id')
        ).order_by('-count')[:5],
        'avg_score': checks.aggregate(Avg('suitability_score'))['suitability_score__avg'] or 0,
        'avg_cost': checks.aggregate(Avg('estimated_cost'))['estimated_cost__avg'] or 0,
        'avg_time': checks.aggregate(Avg('estimated_time_days'))['estimated_time_days__avg'] or 0,
    }

    context = {
        'stats': stats
    }

    return render(request, 'locations/statistics.html', context)